import * as crypto from 'crypto';

/**
 * Derives key and IV using OpenSSL EVP_BytesToKey method (MD5 hash-based).
 */
function evpBytesToKey(password: string, salt: Buffer): { key: Buffer; iv: Buffer } {
  let data = Buffer.alloc(0);
  let d = Buffer.alloc(0);

  while (data.length < 48) {
    d = crypto.createHash('md5').update(Buffer.concat([d, Buffer.from(password), salt])).digest();
    data = Buffer.concat([data, d]);
  }

  return {
    key: data.slice(0, 32), // AES-256 key
    iv: data.slice(32, 48), // 16-byte IV
  };
}

/**
 * Encrypts a JSON-compatible payload using AES-256-CBC with salt.
 */
export function payWiseEncrypt(secretKey: string, value: unknown): string {
  const salt = crypto.randomBytes(8);
  const { key, iv } = evpBytesToKey(secretKey, salt);

  const json = JSON.stringify(value);
  console.log('[EncryptionService] Encrypting payload:', json);

  const cipher = crypto.createCipheriv('aes-256-cbc', key, iv);
  let ciphertext = cipher.update(json, 'utf8', 'base64');
  ciphertext += cipher.final('base64');

  const envelope = {
    ct: ciphertext,
    iv: iv.toString('hex'),
    s: salt.toString('hex'),
  };

  return Buffer.from(JSON.stringify(envelope), 'utf8').toString('base64');
}

/**
 * Decrypts a PayWise-compatible AES-256-CBC payload.
 */
export function payWiseDecrypt<T = unknown>(secretKey: string, encryptedBase64: string): T {
  interface EncryptedEnvelope {
    ct: string;
    iv: string;
    s: string;
  }

  const decodedJson = Buffer.from(encryptedBase64, 'base64').toString('utf8');

  let envelope: EncryptedEnvelope;
  try {
    const parsed = JSON.parse(decodedJson);
    if (
      typeof parsed !== 'object' ||
      parsed === null ||
      typeof parsed.ct !== 'string' ||
      typeof parsed.iv !== 'string' ||
      typeof parsed.s !== 'string'
    ) {
      throw new Error('Envelope is missing required fields or types are invalid');
    }
    envelope = parsed;
  } catch (err) {
    throw new Error(`Invalid encrypted payload: ${(err as Error).message}`);
  }

  const salt = Buffer.from(envelope.s, 'hex');
  const iv = Buffer.from(envelope.iv, 'hex');
  const ct = Buffer.from(envelope.ct, 'base64');

  const { key } = evpBytesToKey(secretKey, salt);

  const decipher = crypto.createDecipheriv('aes-256-cbc', key, iv);
  let decrypted = decipher.update(ct, undefined, 'utf8');
  decrypted += decipher.final('utf8');

  try {
    return JSON.parse(decrypted) as T;
  } catch {
    return decrypted as unknown as T;
  }
}
