using System.Security.Cryptography;
using System.Text;
using System.Text.Json;
using System.Text.Json.Serialization;

namespace PayWiseDotNet;

public static class PayWiseCrypto
{
    private sealed class Envelope
    {
        [JsonPropertyName("ct")]
        public string Ct { get; set; } = string.Empty;

        [JsonPropertyName("iv")]
        public string Iv { get; set; } = string.Empty;

        [JsonPropertyName("s")]
        public string S { get; set; } = string.Empty;
    }

    public static string EncryptPayWise(string secretKey, object value)
    {
        if (string.IsNullOrWhiteSpace(secretKey))
        {
            throw new ArgumentException("Secret key is required.", nameof(secretKey));
        }

        byte[] salt = RandomNumberGenerator.GetBytes(8);
        (byte[] key, byte[] iv) = EvpBytesToKey(secretKey, salt);

        string json = JsonSerializer.Serialize(value);
        byte[] plainBytes = Encoding.UTF8.GetBytes(json);

        byte[] cipherBytes;
        using (var aes = Aes.Create())
        {
            aes.KeySize = 256;
            aes.BlockSize = 128;
            aes.Mode = CipherMode.CBC;
            aes.Padding = PaddingMode.PKCS7;
            aes.Key = key;
            aes.IV = iv;

            using ICryptoTransform encryptor = aes.CreateEncryptor();
            cipherBytes = encryptor.TransformFinalBlock(plainBytes, 0, plainBytes.Length);
        }

        var envelope = new Envelope
        {
            Ct = Convert.ToBase64String(cipherBytes),
            Iv = Convert.ToHexString(iv).ToLowerInvariant(),
            S = Convert.ToHexString(salt).ToLowerInvariant(),
        };

        string envelopeJson = JsonSerializer.Serialize(envelope);
        return Convert.ToBase64String(Encoding.UTF8.GetBytes(envelopeJson));
    }

    public static T DecryptPayWise<T>(string secretKey, string encryptedBase64)
    {
        if (string.IsNullOrWhiteSpace(secretKey))
        {
            throw new ArgumentException("Secret key is required.", nameof(secretKey));
        }

        if (string.IsNullOrWhiteSpace(encryptedBase64))
        {
            throw new ArgumentException("Encrypted payload is required.", nameof(encryptedBase64));
        }

        string decodedJson = Encoding.UTF8.GetString(Convert.FromBase64String(encryptedBase64));
        using JsonDocument envelopeDoc = JsonDocument.Parse(decodedJson);
        JsonElement root = envelopeDoc.RootElement;

        if (!root.TryGetProperty("ct", out JsonElement ctElem) ||
            !root.TryGetProperty("iv", out JsonElement ivElem) ||
            !root.TryGetProperty("s", out JsonElement sElem))
        {
            throw new InvalidOperationException("Invalid encrypted payload envelope.");
        }

        string ctBase64 = ctElem.GetString() ?? throw new InvalidOperationException("Missing ct value.");
        string ivHex = ivElem.GetString() ?? throw new InvalidOperationException("Missing iv value.");
        string saltHex = sElem.GetString() ?? throw new InvalidOperationException("Missing s value.");

        byte[] ct = Convert.FromBase64String(ctBase64);
        byte[] iv = Convert.FromHexString(ivHex);
        byte[] salt = Convert.FromHexString(saltHex);

        (byte[] key, _) = EvpBytesToKey(secretKey, salt);

        byte[] decryptedBytes;
        using (var aes = Aes.Create())
        {
            aes.KeySize = 256;
            aes.BlockSize = 128;
            aes.Mode = CipherMode.CBC;
            aes.Padding = PaddingMode.PKCS7;
            aes.Key = key;
            aes.IV = iv;

            using ICryptoTransform decryptor = aes.CreateDecryptor();
            decryptedBytes = decryptor.TransformFinalBlock(ct, 0, ct.Length);
        }

        string decryptedJson = Encoding.UTF8.GetString(decryptedBytes);
        T? deserialized = JsonSerializer.Deserialize<T>(decryptedJson);

        if (deserialized is null)
        {
            throw new InvalidOperationException("Decrypted payload could not be deserialized.");
        }

        return deserialized;
    }

    private static (byte[] key, byte[] iv) EvpBytesToKey(string password, byte[] salt, int keyLen = 32, int ivLen = 16)
    {
        byte[] passwordBytes = Encoding.UTF8.GetBytes(password);
        int totalLen = keyLen + ivLen;

        byte[] derived = Array.Empty<byte>();
        byte[] block = Array.Empty<byte>();

        while (derived.Length < totalLen)
        {
            byte[] input = new byte[block.Length + passwordBytes.Length + salt.Length];
            Buffer.BlockCopy(block, 0, input, 0, block.Length);
            Buffer.BlockCopy(passwordBytes, 0, input, block.Length, passwordBytes.Length);
            Buffer.BlockCopy(salt, 0, input, block.Length + passwordBytes.Length, salt.Length);

            block = MD5.HashData(input);

            byte[] nextDerived = new byte[derived.Length + block.Length];
            Buffer.BlockCopy(derived, 0, nextDerived, 0, derived.Length);
            Buffer.BlockCopy(block, 0, nextDerived, derived.Length, block.Length);
            derived = nextDerived;
        }

        byte[] key = new byte[keyLen];
        byte[] iv = new byte[ivLen];
        Buffer.BlockCopy(derived, 0, key, 0, keyLen);
        Buffer.BlockCopy(derived, keyLen, iv, 0, ivLen);
        return (key, iv);
    }
}
